
import { StandardLabel } from '/sd:standard-label.js'; 
import { ThemeClass } from '/sd:theme-class.js'; 

/**
 * class to create instance of Selectable Field. It allows for define field which reacts on Left Mouse Button ("onclick" attribute), and Right Mouse Button ("oncontextmenu" attribute). The field change class if it is selected.
 */
export class SelectableField extends StandardLabel {	
	/**
	 * Constructor
	 * @param    {jQuery} xmlObject    			Data loaded from XML
	 * @param    {String} parentId    			HTML ID of the object, which has to be a parent of constructing object
	 * @param    {String} fieldMainClass    	Name of the main CSS style's class
	 * @param	 {String} lmbClickAction		This value is insert inside of onclck="" attribute of HTML field (lmb prefix means Left Mouse Button)
	 * @param	 {String} rmbClickAction		This value is insert inside of oncontextmenu="" attribute of HTML field (rmb prefix means Right Mouse Button)
	 * @param	 {String} selectFunction		This value is insert inside of oncontextmenu="" and onlick="" function supporting unselecting and selecting field
	 * @param    {Integer} itemNumber   		Number of object constructing item - if it is null, it means, that item is not numerable
	 * @param    {String} objId    				ID of creating object - if itemNumber is not null, then itemNumber will be suffix of the ID
	 * @param    {Boolean} selected   			It specifies initial state of the field
	 */
	constructor(xmlObject, parentId, fieldMainClass, lmbClickAction = null, rmbClickAction = null, selectFunction = null, itemNumber = null, objId = null, selected = false) {
		super(xmlObject, parentId, fieldMainClass, itemNumber, objId);
		if(this.parentId != null) {
			this.mainObj.setAttribute("onclick", (lmbClickAction != null) ? lmbClickAction + ";" + selectFunction + ";" : selectFunction + ";");
			this.mainObj.setAttribute("oncontextmenu", (rmbClickAction != null) ? rmbClickAction + ";" + selectFunction + ";" : selectFunction + ";");
			this.selected = selected;
			if(selected)
				this.selectField();
		}
	}
	
	/**
	 * Creates object in HTML file - adjusted to work with SelectableField
	 * @override Method in BasicContainer
	 * @param {NamedNodeMap} attributes List of attributes of the HTML object loaded from XML
	 */
	createObject(attributes) {
		this.mainObj = document.createElement("div");
		this.mainObj.setAttribute("id", this.id);
		this.parentObj.appendChild(this.mainObj);
		
		if(this.itemNumber % 2 == 0) {
			this.defaultStylesTable = window.theme.getStyleArray(this.id, "odd");
			this.selectedStylesTable = window.theme.getStyleArray(this.id, "selectedOdd");
		}
		else {
			this.defaultStylesTable = window.theme.getStyleArray(this.id, "even");
			this.selectedStylesTable = window.theme.getStyleArray(this.id, "selectedEven");
		}
		if(this.selectedStylesTable == null) {
			this.selectedStylesTable = window.theme.getStyleArray(this.id, "selected");
		}

		this.createThemeSchema();
		if(attributes != null)
			this.readAllStylesFromAttributes(attributes);

		if(this.selected)
			this.selectField();
	}

	/**
	* Ivokes request to ThemeClass object instance to load styles defined in XML file to the instance of SelectableField
	* @override Method in BasicContainer
	*/
	createThemeSchema() {
		var element = document.getElementById(this.id);
		if(this.defaultStylesTable != null) {
			ThemeClass.setTableAsStyle(element, this.defaultStylesTable);
		}
	}

	/**
	 * Gets field ID
	 * @returns {String} ID
	 */
	getId() {
		return this.id;
	}
	
	/**
	 * Gets label text of the field
	 * @returns {String}	Label text
	 */
	getLabelText() {
		return "";
	}

	/**
	 * Gets field selectrion status
	 * @returns {Boolean} True if field is selected false if it isn't
	 */
	getSelectionStatus() {
		return this.selected;
	}

	/**
	 * Selects the field and changes CSS class to selected class
	 */
	selectField() {
		var element = document.getElementById(this.id);
		if(this.selectedStylesTable != null) {
			ThemeClass.setTableAsStyle(element, this.selectedStylesTable);
		}
		this.selected = true;
	}

	/**
	 * Unselects the field and changes CSS class to main class
	 */
	unselectField() {
		this.createThemeSchema();
		this.selected = false;
	}

}